<?php
/**
 * 2007-2021 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 * @author    Wirtualna Polska Media S.A.
 * @copyright 2021-2022 Wirtualna Polska Media S.A.
 *  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of PrestaShop SA
 */

require_once(dirname(__FILE__).'/../../config/config.inc.php');

if (!defined('_PS_VERSION_')) {
    exit;
}

class WpPixelPrestashop extends Module
{
    /**
     * @var string Add To Cart classes, ids for buttons
     */
    private $addtocartDefault = '#layer_cart, #blockcart-modal';
    private $trackerInjected = false;

    /**
     * @var boolean
     */
    private $isSuccessSaveForm;

    public function __construct()
    {
        $this->name = 'wppixelprestashop';
        $this->tab = 'analytics_stats';
        $this->version = '1.0.8';
        $this->author = 'Wirtualna Polska Media S.A.';
        $this->need_instance = 0;

        $this->module_key = '93c2e9bc9e3841d18af9bf88a6644cbc';

        /**
         * Set $this->bootstrap to true if your module is compliant with bootstrap (PrestaShop 1.6)
         */
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Wirtualna Polska Pixel');
        $this->description = $this->l('A tool for measuring actions taken by users visiting the website and increasing the effectiveness of advertising on the WP Advertising Network.');

        $this->ps_versions_compliancy = array('min' => '1.6', 'max' => _PS_VERSION_);
    }

    /**
     * Don't forget to create update methods if needed:
     * http://doc.prestashop.com/display/PS16/Enabling+the+Auto-Update
     */
    public function install()
    {
        Configuration::updateValue('WP_PRESTASHOP_PIXEL_EAN', 'ean13');
        Configuration::updateValue('WP_PRESTASHOP_PIXEL_CLASS', $this->addtocartDefault);

        return parent::install() &&
            $this->registerHook('header') &&
            $this->registerHook('displayHeader') &&
            $this->registerHook('displayBeforeBodyClosingTag') &&
            $this->registerHook('displayFooterAfter') &&
            $this->registerHook('displayOrderDetail') &&
            $this->registerHook('displayProductButtons') && //displayProductAdditionalInfo
            $this->registerHook('actionCartSave');
    }

    public function uninstall()
    {
        Configuration::deleteByName('WP_PRESTASHOP_PIXEL_ID');
        Configuration::deleteByName('WP_PRESTASHOP_PIXEL_EAN');
        Configuration::deleteByName('WP_PRESTASHOP_PIXEL_SIZE');
        Configuration::deleteByName('WP_PRESTASHOP_PIXEL_COLOR');
        Configuration::deleteByName('WP_PRESTASHOP_PIXEL_CLASS');

        return parent::uninstall();
    }

    /**
     * Load the configuration form
     */
    public function getContent()
    {
        /**
         * If values have been submitted in the form, process.
         */
        if (((bool)Tools::isSubmit('submitWp_pixel_prestashopModule')) == true) {
            $this->postProcess();
        }

        return $this->renderForm();
    }

    /**
     * Create the form that will be displayed in the configuration of your module.
     */
    protected function renderForm()
    {
        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitWp_pixel_prestashopModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            .'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValues(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($this->getConfigForm()));
    }

    /**
     * Create the structure of your form.
     */
    protected function getConfigForm()
    {
        $id_lang = $this->context->language->id;
        $id_shop = $this->context->shop->id;

        $features = array(
            array(
                'id_attribute_group' => '',
                'name' => $this->l('Attributes not exists'),
            ),
        );
        $features = array_merge($features, $this->getShopFeatures($id_lang, $id_shop));

        $form = array(
            'form' => array(
                'legend' => array(
                    'title' => sprintf('%s %s', $this->l('Wirtualna Polska Pixel'), $this->l('Settings')),
                    'icon' => 'icon-cogs',
                ),
                'input' => array(
                    array(
                        'col' => 5,
                        'type' => 'text',
                        'name' => 'WP_PRESTASHOP_PIXEL_ID',
                        'label' => $this->l('Pixel ID'),
                    ),
                    array(
                        'type' => 'select',
                        'label' => 'EAN',
                        'name' => 'WP_PRESTASHOP_PIXEL_EAN',
                        'desc' => $this->l('Select the place to save the EAN code in the product parameters in your store, default value: ean13'),
                        'multiple' => false,
                        'options' => array(
                            'query' => array(
                                array('key' => 'ean13', 'name' => 'ean13'),
                                array('key' => 'isbn', 'name' => 'isbn'),
                                array('key' => 'upc', 'name' => 'upc'),
                                array('key' => 'mpn', 'name' => 'mpn'),
                            ),
                            'id' => 'key',
                            'name' => 'name'
                        )
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('Size'),
                        'name' => 'WP_PRESTASHOP_PIXEL_SIZE',
                        'desc' => $this->l('Select the product attribute that is responsible for the size in your store, if you do not use sizes, skip the option'),
                        'default_value' => $this->l('Size'),
                        'options' => array(
                            'query' => $features,
                            'id' => 'id_attribute_group',
                            'name' => 'name',
                        ),
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('Color'),
                        'name' => 'WP_PRESTASHOP_PIXEL_COLOR',
                        'desc' => $this->l('Select the product attribute that is responsible for the color in your store, if you do not use colors, skip the option'),
                        'default_value' => $this->l('Color'),
                        'options' => array(
                            'query' => $features,
                            'id' => 'id_attribute_group',
                            'name' => 'name',
                        ),
                    ),
                    array(
                        'col' => 5,
                        'type' => 'text',
                        'name' => 'WP_PRESTASHOP_PIXEL_CLASS',
                        'label' => $this->l('Popup after adding to cart (advanced)'),
                        'desc' => sprintf('%s , . # - _ ', $this->l('Advanced, enter here the classes or IDs of modal box that are displayed after adding a product to the cart. Allowed characters:')),
                        'default_value' => $this->addtocartDefault,
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save'),
                ),
            ),
        );

        if((((bool)Tools::isSubmit('submitWp_pixel_prestashopModule')) == true)) {
            if($this->isSuccessSaveForm) {
                $form['form']['success'] = $this->l('Changes saved!');
            } else {
                $form['form']['error'] = $this->l('Oops, somethink wrong.');
            }
        }

        return $form;
    }

    /**
     * Set values for the inputs.
     */
    protected function getConfigFormValues()
    {
        return array(
            'WP_PRESTASHOP_LIVE_MODE' => Configuration::get('WP_PRESTASHOP_LIVE_MODE', true),
            'WP_PRESTASHOP_PIXEL_ID' => Configuration::get('WP_PRESTASHOP_PIXEL_ID', ''),
            'WP_PRESTASHOP_PIXEL_EAN' => Configuration::get('WP_PRESTASHOP_PIXEL_EAN', 'ean13'),
            'WP_PRESTASHOP_PIXEL_SIZE' => Configuration::get('WP_PRESTASHOP_PIXEL_SIZE', ''),
            'WP_PRESTASHOP_PIXEL_COLOR' => Configuration::get('WP_PRESTASHOP_PIXEL_COLOR', ''),
            'WP_PRESTASHOP_PIXEL_CLASS' => Configuration::get('WP_PRESTASHOP_PIXEL_CLASS', $this->addtocartDefault),
        );
    }


    /**
     * Save form data.
     */
    protected function postProcess()
    {
        $form_values = $this->getConfigFormValues();

        $errors = [];

        foreach (array_keys($form_values) as $key) {
            if ($key == 'WP_PRESTASHOP_PIXEL_CLASS') {
                $saveValue = $this->parsePixelClass(Tools::getValue($key));
            } else {
                $saveValue = Tools::getValue($key);
            }

            $results = Configuration::updateValue($key, $saveValue);

            if(!$results) {
                $errors[] = $this->l('Error in ') . $key;
            }
        }

        if(is_array($errors) && count($errors)) {
            $this->isSuccessSaveForm = false;
        } else {
            $this->isSuccessSaveForm = true;
        }
    }

    /**
     * Parse - Allowed special characters: # . , _ -
     */
    protected function parsePixelClass($pixel_class)
    {
        return preg_replace("/[^A-Za-z0-9\s#.,_-]+/", "", $pixel_class);
    }


    /**
     * Add the CSS & JavaScript files you want to be added on the FO.
     */
    public function hookDisplayHeader()
    {
        $pixel_id = Configuration::get('WP_PRESTASHOP_PIXEL_ID');
        $pixel_class = Configuration::get('WP_PRESTASHOP_PIXEL_CLASS');

        if (empty($pixel_class)) {
            $pixel_class = $this->addtocartDefault;
        }

        $pixel_versionPrestashop = substr(_PS_VERSION_, 0, 3);

        $this->context->smarty->assign('pixel_id', $pixel_id);
        $this->context->smarty->assign('pixel_class', $pixel_class);
        $this->context->smarty->assign('pixel_versionPrestashop', $pixel_versionPrestashop);
        $this->context->smarty->assign('pixel_versionPlugin', $this->version);


        return $this->display(__FILE__, '/views/templates/front/head.tpl');
    }

    public function hookPrzelewy24DisplayOrderDetail() {
        $orderId = Tools::getValue('id_order');
        $cartId = Tools::getValue('id_cart');

        if (!$orderId && $cartId && ($cardOrderId = Order::getIdByCartId($cartId))) {
           $orderId = $cardOrderId;
        }

        if (!$orderId) {
            // last created order
            $sql = new DbQuery();
            $sql->select('max(id_order) as id');
            $sql->from('orders');
            $sql->where('id_customer = \'' . pSQL(Context::getContext()->customer->id) . '\'');

            $lastInsert = Db::getInstance()->getRow($sql->build());

            if (is_array($lastInsert)) {
                $orderId = $lastInsert['id'];
            }
        }

        if ($orderId)
            return $this->hookDisplayOrderDetail($orderId);

    }

    /**
     * WPH Purchase
     * Prestashop 1.6 - orderconfirmation
     */
    public function hookDisplayOrderDetail($orderId = null)
    {
        $page = Tools::getValue('controller');

        if($page == 'orderdetail' || $page === 'guesttracking') {
            return false;
        }

        $id_order = $orderId ? $orderId : (int)Tools::getValue('id_order');

        $pixel_ean = Configuration::get('WP_PRESTASHOP_PIXEL_EAN');
        $pixel_size = Configuration::get('WP_PRESTASHOP_PIXEL_SIZE');
        $pixel_color = Configuration::get('WP_PRESTASHOP_PIXEL_COLOR');

        $order = new Order($id_order);

        $products = $order->getProducts();

        $contents = [];

        foreach ($products as $product) {
            $product_obj = new Product($product['id_product']);
            $category = new Category($product['id_category_default'], Context::getContext()->language->id);

            $combinations = $this->getCurrentProductCombinations($product_obj, $product['product_attribute_id']);

            $product_sizes = [];
            $product_colors = [];

            if (count($combinations)) {
                foreach ($combinations as $combination) {
                    if ($combination['id_attribute_group'] == $pixel_size) {
                        $product_sizes = [$combination['attribute_name']];
                    } elseif ($combination['id_attribute_group'] == $pixel_color) {
                        $product_colors = [$combination['attribute_name']];
                    }
                }
            }

            $productDetails = $this->getProductWrapper($product_obj);

            $temp_contents = [
                'id'=>$product['id_product'],
                'name'=> $productDetails['name'],
                'category'=>$category->name,
                'price'=>$this->preparePrice(
                    Product::getPriceStatic(
                        $product['id_product'],
                        true,
                        $product['product_attribute_id']
                    )
                ),
                'quantity'=>$product['product_quantity'],
                'in_stock'=>'true',
            ];

            if ($product[$pixel_ean]) {
                $temp_contents['ean'] = $product[$pixel_ean];
            }

            if (is_array($product_sizes) && count($product_sizes)) {
                $temp_contents['sizes'] = $this->parseOneSizeInSize($product_sizes);
            } else {
                $temp_contents['sizes'] = ['onesize'];
            }

            if (is_array($product_colors) && count($product_colors)) {
                $temp_contents['colours'] = $product_colors;
            } else {
                $temp_contents['colours'] = ['onecolour'];
            }

            $contents[] = $temp_contents;
        }

        $value = $this->preparePrice($order->total_products);
        $value_gross = $this->preparePrice($order->total_products_wt);
        $shipping_cost = $this->preparePrice($order->total_shipping_tax_incl);

        $this->context->smarty->assign('currency', $this->getCurrency());
        $this->context->smarty->assign('order_id', $order->reference);
        $this->context->smarty->assign('value', $value);
        $this->context->smarty->assign('value_gross', $value_gross);
        $this->context->smarty->assign('shipping_cost', $shipping_cost);
        $this->context->smarty->assign('contents', $contents);


        return $this->display(__FILE__, '/views/templates/front/purchase.tpl');
    }

    public function injectTrackEvent()
    {
        if($this->trackerInjected) return;

        $page = Tools::getValue('controller');
        echo '<!--';
        var_export($page);
        echo '-->';
        $this->trackerInjected = true;

        if ($page == 'category' || $page == 'search') {
            return $this->wphCategory();
        } elseif ($page == 'order') {
            return $this->wphStartOrder();
        } elseif ($page == 'orderconfirmation') {
            return $this->hookDisplayOrderDetail();
        } elseif($page == 'paymentSuccessful') {
             return $this->hookPrzelewy24DisplayOrderDetail();
        }

    }

    /**
     * All tracks inject to Footer
     */
    public function hookDisplayBeforeBodyClosingTag()
    {
        return $this->injectTrackEvent();
    }

    public function hookDisplayFooterAfter()
    {
        return $this->injectTrackEvent();
    }

    /**
     * WPH Add To Cart
     */
    public function hookActionCartSave()
    {
        if (!$this->active || !Validate::isLoadedObject($this->context->cart) || !Tools::getIsset('id_product')) {
            return false;
        }

        $add = Tools::getValue('add');

        if ($add) {
            $pixel_ean = Configuration::get('WP_PRESTASHOP_PIXEL_EAN');
            $pixel_size = Configuration::get('WP_PRESTASHOP_PIXEL_SIZE');
            $pixel_color = Configuration::get('WP_PRESTASHOP_PIXEL_COLOR');

            $product_id = Tools::getValue('id_product');
            $group_id = Tools::getValue('group');
            $ipa = Tools::getValue('ipa'); // prestashop 1.6

            $product_obj = new Product($product_id, false, (int)$this->context->language->id);

            $id_default_combination = $product_obj->getWsDefaultCombination();

            if (method_exists('Product', 'getIdProductAttributeByIdAttributes') && $group_id) {
                $id_product_attribute = (int)$product_obj->getIdProductAttributeByIdAttributes(
                    $product_id,
                    Tools::getValue('group')
                );
            } elseif (!$group_id && $ipa) {
                $id_product_attribute = $ipa;
            } elseif ($id_default_combination) {
                $id_product_attribute = $id_default_combination;
            } else {
                $id_product_attribute = null;
            }

            $price = $this->preparePrice(Product::getPriceStatic($product_id, true, $id_product_attribute));

            $category = new Category((int)$product_obj->id_category_default, (int)$this->context->language->id);

            $wph_object = [
                'id' => $product_id,
                'name' => $product_obj->name,
                'price' => $price,
                'category' => $category->name,
                'quantity' => Tools::getValue('qty'),
                'in_stock' => true,
            ];

            if ($product_obj->{$pixel_ean}) {
                $wph_object['ean'] = $product_obj->{$pixel_ean};
            }

            $product_sizes = [];
            $product_colors = [];

            if ($id_product_attribute) {
                $combinations = $this->getCurrentProductCombinations($product_obj, $id_product_attribute);

                if (count($combinations)) {
                    foreach ($combinations as $combination) {
                        if ($combination['id_attribute_group'] == $pixel_size) {
                            $product_sizes[] = $combination['attribute_name'];
                        } elseif ($combination['id_attribute_group'] == $pixel_color) {
                            $product_colors[] = $combination['attribute_name'];
                        }
                    }
                }
            }

            if (is_array($product_sizes) && count($product_sizes)) {
                $wph_object['sizes'] = $this->parseOneSizeInSize($product_sizes);
            } else {
                $wph_object['sizes'] = ['onesize'];
            }

            if (is_array($product_colors) && count($product_colors)) {
                $wph_object['colours'] = $product_colors;
            } else {
                $wph_object['colours'] = ['onecolour'];
            }

            $cookie_value = json_encode($wph_object);

            setcookie("wp_add_to_cart", $cookie_value, time() + 3600, "/");
        }
    }

    private function getCurrency() {
        $currency = Context::getContext()->currency;

        return $currency->iso_code;
    }

    /**
     * Product Page and Quick View Product
     */
    public function hookDisplayProductButtons()
    {
        $action = Tools::getValue('action');

        if ($action && $action != 'quickview') {
            return false;
        }

        return $this->wphProduct();
    }

    /**
     * WPH ViewProduct
     */
    private function wphProduct()
    {
        $product_id = (int)Tools::getValue('id_product');
        $id_product_attribute = (int)Tools::getValue('id_product_attribute');

        $product_obj = new Product($product_id, (int)$this->context->language->id, $id_product_attribute);

        if (Validate::isLoadedObject($product_obj)) {
            $product = $this->getProductWrapper($product_obj, $id_product_attribute);

            $product_price = Product::getPriceStatic($product_id, true, $id_product_attribute);
            $product_price = $this->preparePrice($product_price);

            $product_quantity = 1;

            $product_in_stock = ($product['in_stock']) ? 'true' : 'false';

            $sizes = $this->parseOneSizeInSize($product['sizes']);

            $this->context->smarty->assign('currency', $this->getCurrency());
            $this->context->smarty->assign('product_id', $product_id);
            $this->context->smarty->assign('product_category', $product['category']);
            $this->context->smarty->assign('product_name', $product['name']);
            $this->context->smarty->assign('product_ean', $product['ean']);
            $this->context->smarty->assign('product_price', $product_price);
            $this->context->smarty->assign('product_quantity', $product_quantity);
            $this->context->smarty->assign('product_in_stock', $product_in_stock);

            $this->context->smarty->assign('product_sizes', $sizes);
            $this->context->smarty->assign('product_colors', $product['colors']);

            return $this->display(__FILE__, '/views/templates/front/product.tpl');
        }
    }

    /**
     * WPH ProductList
     */
    private function wphCategory()
    {
        $id_category = (int)Tools::getValue('id_category');

        $category = new Category($id_category, Context::getContext()->language->id);

        // get product list from smarty variable created in ProductListingFrontController.php
        $listing = (array) $this->context->smarty->getTemplateVars('listing');

        if (is_array($listing) && !count($listing)) {
            $products = (array) $this->context->smarty->getTemplateVars('products');
        } else {
            $products = $listing['result']->getProducts();
        }

        $contents = [];

        if (is_array($products) && count($products)) {
            foreach ($products as $product) {
                $product_obj = new Product($product['id_product'], false, Context::getContext()->language->id);

                $product = $this->getProductWrapper($product_obj);

                $product_in_stock = ($product['in_stock']) ? 'true' : 'false';

                $sizes = $this->parseOneSizeInSize($product['sizes']);

                $temp_contents = [
                    'id' => $product['id_product'],
                    'name' => $product['name'],
                    'price' => $this->preparePrice(Product::getPriceStatic($product['id_product'])),
                    'ean' => $product['ean'],
                    'category' => $product['category'],
                    'sizes' => $sizes,
                    'colours' => $product['colors'],
                    'quantity' => 1,
                    'in_stock' => $product_in_stock
                ];

                $contents[] = $temp_contents;
            }
        }

        $this->context->smarty->assign('currency', $this->getCurrency());
        $this->context->smarty->assign('category_name', $category->name);
        $this->context->smarty->assign('category_contents', $contents);

        return $this->display(__FILE__, '/views/templates/front/category.tpl');
    }

    /**
     * WPH StartOrder
     */
    private function wphStartOrder()
    {
        $products = $this->context->cart->getProducts();
        if (is_array($products) && count($products)) {
            foreach ($products as $product) {
                $product_obj = new Product($product['id_product'], false, Context::getContext()->language->id);

                $productDetails = $this->getProductWrapper($product_obj);
                $sizes = $this->parseOneSizeInSize($productDetails['sizes']);

                $temp_contents = [
                    'id' => $product['id_product'],
                    'name' => $productDetails['name'],
                    'price' => $this->preparePrice(Product::getPriceStatic($product['id_product'])),
                    'ean' => $productDetails['ean'],
                    'category' => $productDetails['category'],
                    'sizes' => $sizes,
                    'colours' => $productDetails['colors'],
                    'quantity' => $product['cart_quantity'],
                ];

                $contents[] = $temp_contents;
            }
        }

        $this->context->smarty->assign('currency', $this->getCurrency());
        $this->context->smarty->assign('contents', $contents);

        return $this->display(__FILE__, '/views/templates/front/startOrder.tpl');
    }

    /**
     * Get Current Product Combination
     */
    private function getCurrentProductCombinations($product_obj, $id_product_attribute)
    {
        $currentCombinations = [];

        if (method_exists('Product', 'getAttributeCombinationsById')) {
            $currentCombinations = $product_obj->getAttributeCombinationsById(
                $id_product_attribute,
                Context::getContext()->language->id
            );
        } else {
            $combinations = $product_obj->getAttributeCombinations(Context::getContext()->language->id);

            if (is_array($combinations) && count($combinations)) {
                foreach ($combinations as $combination) {
                    if ($combination['id_product_attribute'] == $id_product_attribute) {
                        $currentCombinations[] = $combination;
                    }
                }
            }
        }

        return $currentCombinations;
    }

    /**
     * Get product data from Combinations
     */
    private function getDataFromCombinations($product_obj, $id_product_attribute = null)
    {
        $pixel_ean = Configuration::get('WP_PRESTASHOP_PIXEL_EAN');
        $pixel_size = Configuration::get('WP_PRESTASHOP_PIXEL_SIZE');
        $pixel_color = Configuration::get('WP_PRESTASHOP_PIXEL_COLOR');

        $product_sizes = [];
        $product_colors = [];
        $product_ean = $product_obj->{$pixel_ean};

        $combinations = $product_obj->getAttributeCombinations((int)$this->context->language->id);

        if (count($combinations)) {
            $found = false;

            foreach ($combinations as $combination) {
                if (!$found &&
                    $id_product_attribute !== null &&
                    $combination['id_product_attribute'] == $id_product_attribute
                ) {
                    if ($combination[$pixel_ean]) {
                        $product_ean = $combination[$pixel_ean];
                    }

                    $found = true;
                }

                if ($combination['id_attribute_group'] == $pixel_size &&
                    !in_array($combination['attribute_name'], $product_sizes)
                ) {
                    $product_sizes[] = $combination['attribute_name'];
                } elseif ($combination['id_attribute_group'] == $pixel_color &&
                    !in_array($combination['attribute_name'], $product_colors)
                ) {
                    $product_colors[] = $combination['attribute_name'];
                }
            }
        }

        $product_sizes = $this->parseOneSizeInSize($product_sizes);

        if (!count($product_sizes)) {
            $product_sizes = ['onesize'];
        }

        if (!count($product_colors)) {
            $product_colors = ['onecolour'];
        }

        return [
            'sizes' => $product_sizes,
            'colors' => $product_colors,
            'ean' => $product_ean,
        ];
    }

    /**
     * Prepare product data for wphProduct and wphCategory
     */
    private function getProductWrapper($product_obj, $id_product_attribute = null)
    {

        if ($id_product_attribute === null) {
            $id_product_attribute = $product_obj->getWsDefaultCombination();
        }

        if (!$id_product_attribute && !empty($product_obj->cache_default_attribute)) {
            $id_product_attribute = $product_obj->cache_default_attribute;
        }

        $fromCombinations = $this->getDataFromCombinations($product_obj, $id_product_attribute);

        $quantity = StockAvailable::getQuantityAvailableByProduct($product_obj->id, null, $this->context->shop->id);

        $product_name = '';
        if (is_array($product_obj->name)) {
            foreach ($product_obj->name as $idName => $name) {
                if ($name) {
                    $product_name = $name;
                }
            }
        } else {
            $product_name = $product_obj->name;
        }

        $wrapper = [
            'id_product' => $product_obj->id,
            'name' => $product_name,
            'price' => $this->preparePrice(Product::getPriceStatic($product_obj->id)),
            'in_stock' => (bool) $quantity
        ];

        if (!empty($product_obj->id_category_default)) {
            $cat = new Category($product_obj->id_category_default, $this->context->language->id);
            $wrapper['category'] = $cat->name;
        }

        $wrapper = array_merge($wrapper, $fromCombinations);

        return $wrapper;
    }

    /**
     * Get shop feautres for setting size id and color id
     */
    private function getShopFeatures($id_lang, $id_shop)
    {
        return Db::getInstance()->executeS('
			SELECT agl.* FROM '._DB_PREFIX_.'attribute_group_lang agl
			LEFT JOIN '._DB_PREFIX_.'attribute_group_shop ags ON (ags.id_attribute_group = agl.id_attribute_group)
			WHERE agl.id_lang = '.(int) $id_lang.' AND ags.id_shop = '.(int) $id_shop.'
			ORDER BY ags.id_attribute_group ASC');
    }

    /**
     * Set price with decimals
     */
    private function preparePrice($price)
    {
        if (method_exists('Context', 'getComputingPrecision')) {
            $formatted = Tools::ps_round($price, Context::getContext()->getComputingPrecision());
        } else {
            $formatted =$price;
        }

        return  (float)number_format($formatted, 2, '.', '');
    }

    /**
     * @param array|string $size
     * parse size params ex. from 'one size' to onesize
     */
    private function parseOneSizeInSize($sizes)
    {
        if(is_array($sizes)) {
            foreach($sizes as &$size) {
                $sizeString = $this->parseSizeTrimLowerCase($size);

                if($sizeString == 'onesize') {
                    $size = $sizeString;
                }
            }
        } else {
            $sizeString = $this->parseSizeTrimLowerCase($sizes);

            if($sizeString == 'onesize') {
                $sizes = $sizeString;
            }
        }

        return $sizes;
    }

    /**
     * @param string $size
     * parse size string trim and lowercase
     */
    private function parseSizeTrimLowerCase($size)
    {
        $sizeParsed = str_replace(' ', '', $size);
        return strtolower(trim($sizeParsed));
    }
}
